/*************************************************************************
 * DISCLAIMER                                                            *
 * Services performed by FREESCALE in this matter are performed          *
 * AS IS and without any warranty. CUSTOMER retains the final decision   *
 * relative to the total design and functionality of the end product.    *
 * FREESCALE neither guarantees nor will be held liable by CUSTOMER      *
 * for the success of this project. FREESCALE disclaims all warranties,  *
 * express, implied or statutory including, but not limited to,          *
 * implied warranty of merchantability or fitness for a particular       *
 * purpose on any hardware, software ore advise supplied to the project  *
 * by FREESCALE, and or any product resulting from FREESCALE services.   *
 * In no event shall FREESCALE be liable for incidental or consequential *
 * damages arising out of this agreement. CUSTOMER agrees to hold        *
 * FREESCALE harmless against any and all claims demands or actions      *
 * by anyone on account of any damage, or injury, whether commercial,    *
 * contractual, or tortuous, rising directly or indirectly as a result   *
 * of the advise or assistance supplied CUSTOMER in connection with      *
 * product, services or goods supplied under this Agreement.             *
 *************************************************************************/
/*******************************************************************
  Copyright (c) 2011 Freescale Semiconductor
  \file     	boot.c
  \brief    	bootloader
  \author   	Norman Guo
  \version    1.0
  \date     	26/Sep/2011
*********************************************************************/


#include "derivative.h"
#include "doonstack.h"               //include flash utilities
#include "start08.h"

#pragma MESSAGE DISABLE C1404        // Return expected
#pragma MESSAGE DISABLE C1420        // Result of function call is ignored
#pragma MESSAGE DISABLE C1801        // Implicit parameter-declaration
#pragma MESSAGE DISABLE C1805        // Non standard coversion used
#pragma MESSAGE DISABLE C1825        // Indirection to different types
#pragma MESSAGE DISABLE C20000       // Dead code detected
#pragma MESSAGE DISABLE C4000        // Condition always true
#pragma MESSAGE DISABLE C4200        // Other segment than in previous declaration
#pragma MESSAGE DISABLE C5909        // Assignment in condition


#define BUSCLOCK 20000000            // BUS clock in Hertz

#ifdef BUSCLOCK
#if (((BUSCLOCK/200000L)-1)>63)
    #define FCDIVVAL ((FCDIV_PRDIV8_MASK | ((((BUSCLOCK/200000L)/8)-1) & FCDIV_DIV_MASK)))
#else
    #define FCDIVVAL (((BUSCLOCK/200000L)-1) & FCDIV_DIV_MASK)
#endif
#else
    #error "BUSCLOCK macro needs to be defined, value in Hertz"
#endif

#define MCU_RESET() asm DCW $9E00                // illegal opcode to cause reset

// Only one of the SCI port can be defined to 1
#define SCI1  1                                  // using SCI1
#define SCI2  0                                  // using SCI2
#define BOOT_SCI_BUFFER_SIZE 80

// Modbus
// T = 2.29 ms  for 4800bps    3.5T = 8.02ms    4T = 9.17ms 
// T = 1.15 ms  for 9600bps    3.5T = 4.01ms    4T = 4.58ms
//

// NVPROT: FPS7=1,FPS6=1,FPS5=1,FPS4=1,FPS3=1,FPS2=0,FPS1=1,FPDIS=0
// Protect FLASH from 0xFC00 to 0xFFFF
const unsigned char NVPROT_INIT @0x0000FFBD = 0xFA;

// NVOPT: KEYEN=0,FNORED=1,??=1,??=1,??=1,??=1,SEC01=0,SEC00=0
// MCU secured
// const unsigned char NVOPT_INIT @0x0000FFBF = 0x7C;

#pragma CONST_SEG BOOT_CONST
const char Identifier[20] = "MC9S08AC32";        // String ended with 0
const byte const_node_addr = 0x01;               // The address for the node
const word RAMEndP1 = 0x086F + 1;                // The end address of RAM plus 1
#pragma CONST_SEG DEFAULT

#pragma CODE_SEG BOOT_CODE
// Function prototypes
void ISR_FFFE(void);
void boot_start(byte boot_flag);
unsigned char boot_get_irq(void);
void boot_start(byte boot_flag);
void boot_init_pll(void);
void boot_init_sci(void);
byte boot_sci_getchar(byte * timeout);
byte boot_sci_puthar(char Chr);
void boot_init_tpm1(void);
void modbus_send(byte * buffer);
byte get_node_addr(void);

//////////////////////////////////////////////////////////////////////
// The reset vector points here
//////////////////////////////////////////////////////////////////////
void _Boot()
{
  __asm LDHX RAMEndP1;
  __asm TXS; 
  boot_start(0);
}

//////////////////////////////////////////////////////////////////////
// It will jump to user code if all these conditions are true:
// 1. IRQ pin is high
// 2. boot_flag is 0
// 3. Relocated_Reset is not 0xFFFF
//
// otherwise it will enter the bootloader code.
//
// For parameter boot_flag
// If comes from reset set boot_flag = 0.
// If a command 'B' is received in the user code, set boot_flag = 1.
// If a command 'V' is received in the user code, set boot_flag = 2.
// 
//////////////////////////////////////////////////////////////////////
void boot_start(byte boot_flag)
{ 
  
  byte i=0, temp=0, node_addr = 0;
  byte sci_buffer[BOOT_SCI_BUFFER_SIZE],pointer = 0;
  volatile byte timeout = 0;
  dword remapped_reset = 0;
  
  volatile union 
  {
    word address_w;
    struct
    {
      byte addr_hi;
      byte addr_lo;
    }address_b;
  }target_address;
  
  remapped_reset = ((*((dword*)(ISR_FFFE)))>>8)&0x0000FFFF;
      
  FCDIV = FCDIVVAL;                              // initialize Flash clock for 150-200kHz range
  DisableInterrupts;
  IRQSC_IRQPE = 1;
  if(   boot_get_irq()                           // other methods can be added here to force the code to enter bootloader
     && (boot_flag == 0) 
     && (remapped_reset != 0xFFFF))
  {
    _asm JMP ISR_FFFE;                           // Enter user code
  }
  else
  {                                              // Enter bootloader
    boot_init_pll( );
    boot_init_sci( );
    boot_init_tpm1( );
    node_addr = get_node_addr( );
    if((boot_flag == 1) && (remapped_reset != 0xFFFF))
      FlashErase((unsigned char *)(ISR_FFFE));
    
    while(1)
    {
      pointer = 0;
      while(1)
      {
        temp = boot_sci_getchar(&timeout);
        if(timeout)
        {
          if(pointer > 0)
          {
            break;
          }
          pointer = 0;
          timeout = 0;
        }
        else
        {
          sci_buffer[pointer] = temp;
          pointer++;
        }
      }
      
      //pointer received datalength
      // check_CRC();
      
      if(sci_buffer[0] != node_addr || sci_buffer[1] != 0x43)        // check node address and the function code  
        continue;                                                    // 0x43 is for firmware updating
   
      
      if(sci_buffer[3] == 'E') 
      {
        //target_address = (word)(sci_buffer[4]) << 8 + (word)sci_buffer[5];
        target_address.address_b.addr_hi = sci_buffer[4];
        target_address.address_b.addr_lo = sci_buffer[5];
        FlashErase((unsigned char *)(target_address.address_w));
        sci_buffer[2] = 0x00;                                        // datalength
        modbus_send(sci_buffer);
      }
      else if(sci_buffer[3] == 'W') 
      {
        //target_address = (word)(sci_buffer[4]) << 8 + (word)sci_buffer[5];
        target_address.address_b.addr_hi = sci_buffer[4];
        target_address.address_b.addr_lo = sci_buffer[5];
        if (FlashProgBurst((unsigned char *)(target_address.address_w), sci_buffer + 7, sci_buffer[6]))
        {
          sci_buffer[1] = 0xC4;                                      // error functin code
          sci_buffer[2] = 0x04;                                      // only if access or violation errors occur
        }
        else
        {
          sci_buffer[2] = 0x00;                                      // datalength
        }
        modbus_send(sci_buffer);
      }
      else if(sci_buffer[3] == 'R') 
      {
        //target_address = (word)(sci_buffer[4]) << 8 + (word)sci_buffer[5];
        target_address.address_b.addr_hi = sci_buffer[4];
        target_address.address_b.addr_lo = sci_buffer[5];
        temp = sci_buffer[6];                                        // length to be read
        for(i = 0; i < temp; i ++)
          sci_buffer[i + 3] = *(unsigned char *)(target_address.address_w + i);
        sci_buffer[2] = temp;                                        // datalength
        modbus_send(sci_buffer);
      }
      else if(sci_buffer[3] == 'I')
      {
        pointer = 3;
        while(sci_buffer[pointer] = Identifier[pointer -3])
          pointer++;
        sci_buffer[2] = pointer - 2;  
        modbus_send(sci_buffer);
      }
      else if(sci_buffer[3] == 'G')                                  // run the code
      {
        MCU_RESET();                                                 // reset the MCU to run user code
      //_asm JMP ISR_FFFE;                                           // we can also jump to _Startup without reset
      } 
        
      for(pointer = 0;pointer < BOOT_SCI_BUFFER_SIZE;pointer ++)     // clear sci buffer
        sci_buffer[pointer] = 0;
      
    }
    
  }
  __RESET_WATCHDOG();
  while(1);                                                          // Never reached here  
}

//////////////////////////////////////////////////////////////////////
// Function to get the node address of this control unit. It is intended
// for user to get it when address is stored somewhere else. Do not call
// any function allocated in unproteced user code which may be erased
// during firmware updating.
//////////////////////////////////////////////////////////////////////
byte get_node_addr( )
{
  return const_node_addr;
}

void modbus_send(byte * buffer)
{
  byte datalength,pointer;
   
  TPM1CNTH = 0x00;
  (void)(TPM1C1SC);
  TPM1C1SC_CH1F = 0x00;
  while(!TPM1C1SC_CH1F)                                              // wait 4T time
    __RESET_WATCHDOG();
  // process command
  datalength = buffer[2];
  buffer[datalength + 3] = 0xAA;                                     // put CRCH here if CRC is implemented
  buffer[datalength + 4] = 0x55;                                     // put CRCL here if CRC is implemented
  
  datalength = datalength + 5;
  pointer = 0;          
  while(datalength)
  {
    boot_sci_puthar(buffer[pointer]);
    __RESET_WATCHDOG();
    datalength --;
    pointer ++;
  }
  
}

// FEI mode Bus = 20MHz
void boot_init_pll(void)
{
  // System clock initialization
  // ICGC1: HGO=0,RANGE=1,REFS=0,CLKS1=0,CLKS0=1,OSCSTEN=1,LOCD=0
  ICGC1 = 0x4C;                 
  // ICGC2: LOLRE=0,MFD2=1,MFD1=1,MFD0=1,LOCRE=0,RFD2=0,RFD1=0,RFD0=0
  ICGC2 = 0x70;                 
  if (*(unsigned char*)0xFFBE != 0xFF) {                             // Test if the device trim value is stored on the specified address
    ICGTRM = *(unsigned char*)0xFFBE;                                // Initialize ICGTRM register from a non volatile memory
  }
  while(!ICGS1_LOCK) {                                               // Wait
    __RESET_WATCHDOG();
  }
}

unsigned char boot_get_irq(void)
{
  asm {
    CLRA
    BIL  low
    COMA
    low:
  }
}

void boot_init_sci(void)
{
#if SCI1  
  SCI1C1 = 0x13;                     // 1 start, 8 data, odd parity, 1 stop
  SCI1C3 = 0x00;                     // Disable error interrupts
  SCI1S2 = 0x00;                
  SCI1C2 = 0x00;                     // Disable all interrupts
  SCI1BDH = 0x01;                    // 
  SCI1BDL = 0x04;                    // Set baud rate to 4800 when Bus = 20MHz
//SCI1BDH = 0x00;                    // 
//SCI1BDL = 0x0B;                    // Set baud rate to 115200 when Bus = 20MHz
  SCI1C2 = SCI1C2_TE_MASK | SCI1C2_RE_MASK;      // Enable Tx and Rx
#elif SCI2
  SCI2C1 = 0x13;                     // 1 start, 8 data, odd parity, 1 stop
  SCI2C3 = 0x00;                     // Disable error interrupts
  SCI2S2 = 0x00;                
  SCI2C2 = 0x00;                     // Disable all interrupts
  SCI2BDH = 0x01;                    // 
  SCI2BDL = 0x04;                    // Set baud rate to 4800 when Bus = 20MHz
//SCI2BDH = 0x00;                    // 
//SCI2BDL = 0x0B;                    // Set baud rate to 115200 when Bus = 20MHz
  SCI2C2 = SCI2C2_TE_MASK | SCI2C2_RE_MASK;      // Enable Tx and Rx
#endif
}

byte boot_sci_getchar(byte * time_out)
{
   TPM1CNTH = 0;                     // reset counter
#if SCI1
   while(!SCI1S1_RDRF)
   {
     if(TPM1C0SC_CH0F)               // 3.5T reached
     {
       TPM1C0SC_CH0F = 0;            // clear flag
       *time_out = 1;
       break;
     }
     __RESET_WATCHDOG();
   };
   (void) SCI1S1;
   (void) SCI1C3;
   return SCI1D;
#elif SCI2
   while(!SCI2S1_RDRF)
   {
     if(TPM1C0SC_CH0F)               // 3.5T reached
     { 
       TPM1C0SC_CH0F = 0;            // clear flag
       *time_out = 1;
       break;
     }     
     __RESET_WATCHDOG();
   };
   (void) SCI2S1;
   (void)SCI2C3;
   return SCI2D;
#endif   
}

byte boot_sci_puthar(char Chr)
{
#if SCI1
  while(!SCI1S1_TDRE)
    __RESET_WATCHDOG();
  
  SCI1D = (byte)Chr;                 // Store char to the transmitter register
  while(!SCI1S1_TC)
    __RESET_WATCHDOG();
    
  return 0;

#elif SCI2

  while(!SCI2S1_TDRE)
    __RESET_WATCHDOG();
  
  SCI2D = (byte)Chr;                 // Store char to the transmitter register
  while(!SCI2S1_TC)
    __RESET_WATCHDOG();
    
  return 0;

#endif  
}

void boot_init_tpm1( )
{
  TPM1SC = 0x00;
  TPM1MOD = 0x00;
  TPM1C0SC = 0x10;
  TPM1C0V = 0x4E4D;                  // 3.5T at 20MHz 4800 bps
//TPM1C0V = 0x2727;                  // 3.5T at 20MHz 9600 bps
  TPM1C1SC = 0x10;
  TPM1C1V = 0x5988;                  // 4T at 20MHz  4800 bps
//TPM1C1V = 0x2CB7;                  // 4T at 20MHz  9600 bps
  TPM1CNTH = 0x00;
  TPM1SC = 0x0B;
}

#pragma CODE_SEG VECTOR_ROM
void ISR_FFC6(void)                  //28
{
   Unused_service( );
}

void ISR_FFC8(void)                  //27
{
   Unused_service( );
}

void ISR_FFCA(void)                  //26
{
   Unused_service( );
}

void ISR_FFCC(void)                  //25
{
   Unused_service( );
}

void ISR_FFCE(void)                  //24
{
   Unused_service( );
}

void ISR_FFD0(void)                  //23
{
   Unused_service( );
}

void ISR_FFD2(void)                  //22
{
   Unused_service( );
}

void ISR_FFD4(void)                  //21
{
   Unused_service( );
}

void ISR_FFD6(void)                  //20
{
   Unused_service( );
}

void ISR_FFD8(void)                  //19
{
   Unused_service( );
}

void ISR_FFDA(void)                  //18
{
   Unused_service( );
}

void ISR_FFDC(void)                  //17
{
   Unused_service( );
}

void ISR_FFDE(void)                  //16
{
   Unused_service( );
}

void ISR_FFE0(void)                  //15
{
   Unused_service( );
}

void ISR_FFE2(void)                  //14
{
   Vtpm2ovf_service();
}

void ISR_FFE4(void)                  //13
{
   Vtpm2ch1_service();
}

void ISR_FFE6(void)                  //12
{
   Vtpm2ch0_service();
}

void ISR_FFE8(void)                  //11
{
   Unused_service( );
}

void ISR_FFEA(void)                  //10
{
   Unused_service( );
}

void ISR_FFEC(void)                  //9
{
   Unused_service( );
}

void ISR_FFEE(void)                  //8
{
   Unused_service( );
}

void ISR_FFF0(void)                  //7
{
   Unused_service( );
}

void ISR_FFF2(void)                  //6
{
   Unused_service( );
}

void ISR_FFF4(void)                  //5
{
   Unused_service( );
}

void ISR_FFF6(void)                  //4
{
   Unused_service( );
}

void ISR_FFF8(void)                  //3
{
   Unused_service( );
}

void ISR_FFFA(void)                  //2
{
   Virq_service( );
}

void ISR_FFFC(void)                  //1
{
   Unused_service( );
}

void ISR_FFFE(void)                  //0
{
   _Startup( );                      // Enter user code
}

#pragma CODE_SEG DEFAULT

void (* near const _vect[])() @0xFFC6 = { // Interrupt vector table
         ISR_FFC6,
         ISR_FFC8,
         ISR_FFCA,                
         ISR_FFCC,
         ISR_FFCE,
         ISR_FFD0,
         ISR_FFD2,
         ISR_FFD4,               
         ISR_FFD6,
         ISR_FFD8,
         ISR_FFDA,
         ISR_FFDC,
         ISR_FFDE,                
         ISR_FFE0,
         ISR_FFE2,
         ISR_FFE4,
         ISR_FFE6,
         ISR_FFE8,                 
         ISR_FFEA,
         ISR_FFEC,
         ISR_FFEE,
         ISR_FFF0,
         ISR_FFF2,                
         ISR_FFF4,
         ISR_FFF6,
         ISR_FFF8,
         ISR_FFFA,
         ISR_FFFC,         
 //      _Boot                       // It is defined in project.prm
 };
 
 
/*/////////////////interrupt vector table/////////////////
#define Vtpm3ovf                        0x0000FFC6
#define Vtpm3ch1                        0x0000FFC8
#define Vtpm3ch0                        0x0000FFCA
#define Vrti                            0x0000FFCC
#define Viic1                           0x0000FFCE
#define Vadc1                           0x0000FFD0
#define Vkeyboard1                      0x0000FFD2
#define Vsci2tx                         0x0000FFD4
#define Vsci2rx                         0x0000FFD6
#define Vsci2err                        0x0000FFD8
#define Vsci1tx                         0x0000FFDA
#define Vsci1rx                         0x0000FFDC
#define Vsci1err                        0x0000FFDE
#define Vspi1                           0x0000FFE0
#define Vtpm2ovf                        0x0000FFE2
#define Vtpm2ch1                        0x0000FFE4
#define Vtpm2ch0                        0x0000FFE6
#define Vtpm1ovf                        0x0000FFE8
#define VReserved18                     0x0000FFEA
#define VReserved19                     0x0000FFEC
#define Vtpm1ch3                        0x0000FFEE
#define Vtpm1ch2                        0x0000FFF0
#define Vtpm1ch1                        0x0000FFF2
#define Vtpm1ch0                        0x0000FFF4
#define Vicg                            0x0000FFF6
#define Vlvd                            0x0000FFF8
#define Virq                            0x0000FFFA
#define Vswi                            0x0000FFFC
#define Vreset                          0x0000FFFE*/




